/**
 * @license
 * Copyright 2022 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { ALL_ENVS, describeWithFlags } from '../jasmine_util';
import { expectArraysClose } from '../test_util';
describeWithFlags('searchSorted', ALL_ENVS, () => {
    it('test1D', async () => {
        // Tests against numpy generated data.
        const NUMPY_DATA = {
            'left-float32': [
                [
                    -945.2247924804688, -921.8904418945312, -829.9115600585938,
                    -719.2261352539062, -660.3391723632812, -603.7969970703125,
                    -591.0955200195312, -373.1516418457031, -165.39039611816406,
                    -161.61097717285156, 117.37965393066406, 340.9350280761719,
                    370.4389953613281, 384.6452331542969, 601.4891357421875,
                    752.7783203125, 756.23486328125, 756.2850341796875,
                    789.2133178710938, 936.5231323242188
                ],
                [
                    -165.95599365234375, 440.64898681640625, -999.771240234375,
                    -395.3348693847656, -706.4882202148438, -815.3228149414062,
                    -627.4795532226562, -308.8785400390625, -206.46505737304688,
                    77.63346862792969
                ],
                [8, 14, 0, 7, 4, 3, 5, 8, 8, 10]
            ],
            'left-int32': [
                [
                    -961, -893, -793, -739, -706, -576, -468, -439, -424, -412,
                    -104, -16, 148, 178, 357, 399, 496, 578, 817, 977
                ],
                [-803, -157, 915, 66, 383, -368, 373, 669, -963, 500],
                [2, 10, 19, 12, 15, 10, 15, 18, 0, 17]
            ],
            'right-float32': [
                [
                    -725.0505981445312, -721.4473266601562, -669.2916259765625,
                    -460.14422607421875, -304.4682922363281, -302.20330810546875,
                    -204.64633178710938, -143.817626953125, 243.3914337158203,
                    247.34442138671875, 326.88299560546875, 451.9959716796875,
                    501.62420654296875, 501.8848571777344, 614.7825927734375,
                    766.6121826171875, 791.7724609375, 806.8038330078125,
                    855.0171508789062, 929.6801147460938
                ],
                [
                    -795.3311157226562, -171.88803100585938, 388.8003234863281,
                    -171.64146423339844, -900.0930786132812, 71.79280853271484,
                    327.58929443359375, 29.77822494506836, 889.1895141601562,
                    173.11007690429688
                ],
                [0, 7, 11, 7, 0, 8, 11, 8, 19, 8]
            ],
            'right-int32': [
                [
                    -968, -867, -751, -725, -655, -346, -285, 54, 246, 381,
                    393, 423, 507, 510, 771, 817, 846, 858, 865, 994
                ],
                [-770, 898, -100, 156, -183, -525, 806, 147, -994, 234],
                [2, 19, 7, 8, 7, 5, 15, 8, 0, 8]
            ]
        };
        for (const side of ['left', 'right']) {
            for (const dtype of ['float32', 'int32']) {
                // tslint:disable-next-line:no-unnecessary-type-assertion
                const key = `${side}-${dtype}`;
                const [sortedSequence, values, npAns] = NUMPY_DATA[key];
                const result = tf.searchSorted(sortedSequence, values, side);
                expectArraysClose(await result.data(), npAns);
            }
        }
    });
    it('lowerBound2D', async () => {
        for (const dtype of ['float32', 'int32']) {
            const sortedSequence = tf.tensor2d([[0, 3, 9, 9, 10], [1, 2, 3, 4, 5]], undefined, dtype);
            const values = tf.tensor2d([[2, 4, 9], [0, 2, 6]], undefined, dtype);
            const correctAns = [[1, 2, 2], [0, 1, 5]];
            const result = tf.searchSorted(sortedSequence, values, 'left');
            expectArraysClose(await result.data(), correctAns);
        }
    });
    it('upperBound2D', async () => {
        for (const dtype of ['float32', 'int32']) {
            const sortedSequence = tf.tensor2d([[0, 3, 9, 9, 10], [1, 2, 3, 4, 5]], undefined, dtype);
            const values = tf.tensor2d([[2, 4, 9], [0, 2, 6]], undefined, dtype);
            const correctAns = [[1, 2, 4], [0, 2, 5]];
            const result = tf.searchSorted(sortedSequence, values, 'right');
            expectArraysClose(await result.data(), correctAns);
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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